// Copyright 2014 Google Inc. All Rights Reserved.

#include "RadioEndpoint.h"
#include <google/protobuf/repeated_field.h>

void RadioEndpoint::addDiscoveryInfo(ServiceDiscoveryResponse *sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    RadioService* rs = service->mutable_radio_service();

    for (vector<RadioPropertiesStruct>::iterator it = mRadioConfig.radioProperties.begin();
         it != mRadioConfig.radioProperties.end(); ++it) {
        RadioProperties *rp = rs->add_radio_properties();
        rp->set_radio_id(it->radioId);
        rp->set_type(static_cast<RadioType>(it->radioType));
        for (vector<pair<int32_t, int32_t> >::iterator ch_it = it->channelRange.begin();
             ch_it != it->channelRange.end(); ++ch_it) {
            Range *range = rp->add_channel_range();
            range->set_min(ch_it->first);
            range->set_max(ch_it->second);
        }
        for (vector<int32_t>::iterator cs_it = it->channelSpacings.begin();
             cs_it != it->channelSpacings.end(); ++cs_it) {
            rp->add_channel_spacings(*cs_it);
        }
        rp->set_channel_spacing(it->channelSpacing);
        if (it->backgroundTuner >= 0) {
            rp->set_background_tuner(it->backgroundTuner);
        }
        if (it->region >= 0) {
            rp->set_region(static_cast<ItuRegion>(it->region));
        }
        if (it->rds >= 0) {
            rp->set_rds(static_cast<RdsType>(it->rds));
        }
        if (it->afSwitch >= 0) {
            rp->set_af_switch(it->afSwitch);
        }
        if (it->ta >= 0) {
            rp->set_ta(it->ta);
        }
        if (it->trafficService >= 0) {
            rp->set_traffic_service(static_cast<TrafficServiceType>(it->trafficService));
        }
        if (it->audioLoopback >= 0) {
            rp->set_audio_loopback(it->audioLoopback);
        }
        if (it->muteCapability >= 0) {
            rp->set_mute_capability(it->muteCapability);
        }
        if (it->stationPresetsAccess >= 0) {
            rp->set_station_presets_access(it->stationPresetsAccess);
        }
    }
}

int RadioEndpoint::routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*) msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    if (mRadioCallbacks == NULL) {
        return STATUS_INTERNAL_ERROR;
    }

    switch (type) {
        case RADIO_MESSAGE_STEP_CHANNEL_REQUEST: {
            StepChannelRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleStepRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_SEEK_STATION_REQUEST: {
            SeekStationRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleSeekRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_SCAN_STATIONS_REQUEST: {
            ScanStationsRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleScanRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_TUNE_TO_STATION_REQUEST: {
            TuneToStationRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleTuneRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_GET_PROGRAM_LIST_REQUEST: {
            GetProgramListRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleGetProgramListRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_CANCEL_OPERATIONS_REQUEST: {
            CancelRadioOperationsRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleCancelRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_CONFIGURE_CHANNEL_SPACING_REQUEST: {
            ConfigureChannelSpacingRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleConfigChannelSpacingRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_GET_TRAFFIC_UPDATE_REQUEST: {
            GetTrafficUpdateRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleGetTrafficUpdateRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_MUTE_RADIO_REQUEST: {
            MuteRadioRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleOnMuteRadioRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_RADIO_SOURCE_REQUEST: {
            RadioSourceRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleRadioSourceRequest(req);
            }
            break;
        }
        case RADIO_MESSAGE_SELECT_ACTIVE_RADIO_REQUEST: {
            SelectActiveRadioRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleSelectActiveRadioRequest(req);
            }
            break;
        }
    }
    return ret;
}

void RadioEndpoint::onChannelOpened(uint8_t channelId) {
    ProtocolEndpointBase::onChannelOpened(channelId);

    RadioStateStruct radioStateStruct;
    mRadioCallbacks->onGetCurrentRadioState(&radioStateStruct);

    RadioStateNotification radioStateNotif;
    radioStateNotif.set_radio_source_enabled(radioStateStruct.radioSourceEnabled);
    if (radioStateStruct.radioMuted >= 0) {
        radioStateNotif.set_radio_muted(radioStateStruct.radioMuted == 1);
    }
    radioStateNotif.set_active_radio_id(radioStateStruct.activeRadioId);
    populateStationInfo(radioStateStruct.stationInfo, radioStateNotif.mutable_station_info());
    for (vector<StationInfoStruct>::iterator si_it = radioStateStruct.programList.begin();
         si_it != radioStateStruct.programList.end(); ++si_it) {
        populateStationInfo(*si_it, radioStateNotif.add_program_list());
    }
    for (vector<StationPresetListStruct>::iterator spl_it = radioStateStruct.stationPresetLists.begin();
         spl_it != radioStateStruct.stationPresetLists.end(); ++spl_it) {
        StationPresetList *spl = radioStateNotif.add_station_preset_lists();
        spl->set_name(spl_it->name);
        for (vector<int>::iterator type_it = spl_it->restrictedStationTypes.begin();
             type_it != spl_it->restrictedStationTypes.end(); type_it++) {
            spl->add_restricted_station_types(*type_it);
        }
        for (vector<StationPresetStruct>::iterator sp_it = spl_it->presets.begin();
             sp_it != spl_it->presets.end(); sp_it++) {
            StationPreset *sp = spl->add_presets();
            sp->set_type(static_cast<RadioType>(sp_it->type));
            sp->set_channel(sp_it->channel);
            sp->set_sub_channel(sp_it->subChannel);
        }
    }

    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_STATE_NOTIFICATION, radioStateNotif, &buf);
    if (queueOutgoing(buf.raw(), buf.size()) != STATUS_SUCCESS) {
        LOGE("error sending radio state notification on channel opened");
    }
}

void RadioEndpoint::setRadioConfiguration(const RadioConfigurationStruct &config) {
    mRadioConfig = config;
}

void RadioEndpoint::registerCallbacks(const shared_ptr<IRadioCallbacks>& radioCallbacks) {
    mRadioCallbacks = radioCallbacks;
}

int RadioEndpoint::sendRadioSourceResponse(int32_t status, bool enabled) {
    RadioSourceResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_source_enabled(enabled);
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_RADIO_SOURCE_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendTuneToStationResponse(int32_t status, int32_t radioId) {
    TuneToStationResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_TUNE_TO_STATION_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendStepChannelResponse(int32_t status, int32_t radioId) {
    StepChannelResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_STEP_CHANNEL_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendSeekStationResponse(int32_t status, int32_t radioId) {
    SeekStationResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_SEEK_STATION_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendScanStationsResponse(int32_t status, int32_t radioId, bool started) {
    ScanStationsResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    if (status == STATUS_SUCCESS) {
        resp.set_started(started);
    }
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_SCAN_STATIONS_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendRadioStationInfoNotification(int32_t radioId,
                                                    const StationInfoStruct& stationInfo) {
    RadioStationInfoNotification resp;
    resp.set_radio_id(radioId);

    populateStationInfo(stationInfo, resp.mutable_station_info());
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_RADIO_STATION_INFO_NOTIFICATION, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendActiveRadioNotification(int32_t status, int32_t radioId,
                                               const StationInfoStruct &stationInfo) {
    ActiveRadioNotification resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    populateStationInfo(stationInfo, resp.mutable_station_info());
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_ACTIVE_RADIO_NOTIFICATION, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendStationPresetsNotification(
        const vector<StationPresetListStruct> &presetLists) {
    StationPresetsNotification resp;
    for (vector<StationPresetListStruct>::const_iterator list_it = presetLists.begin();
         list_it != presetLists.end(); ++list_it) {
        StationPresetList *list = resp.add_preset_lists();
        list->set_name(list_it->name);
        for (vector<int32_t>::const_iterator type_it = list_it->restrictedStationTypes.begin();
             type_it != list_it->restrictedStationTypes.end(); type_it++) {
            list->add_restricted_station_types(*type_it);
        }

        for (vector<StationPresetStruct>::const_iterator preset_it = list_it->presets.begin();
             preset_it != list_it->presets.end(); preset_it++) {
            StationPreset *preset = list->add_presets();
            preset->set_type(static_cast<RadioType>(preset_it->type));
            preset->set_channel(preset_it->channel);
            preset->set_sub_channel(preset_it->subChannel);
        }
    }
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_STATION_PRESETS_NOTIFICATION, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendGetTrafficUpdateResponse(int32_t status, int radioId,
                                                const vector<TrafficIncidentStruct> &incidents) {
    GetTrafficUpdateResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    for (vector<TrafficIncidentStruct>::const_iterator it = incidents.begin();
         it != incidents.end(); ++it) {
        TrafficIncident *incident = resp.add_incidents();
        incident->set_event_code(it->eventCode);
        incident->set_expected_incident_duration(it->expectedIncidentDuration);
        Location *loc = incident->mutable_location();
        loc->set_longitude(it->longitude);
        loc->set_latitude(it->latitude);
    }
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_GET_TRAFFIC_UPDATE_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendMuteRadioResponse(int32_t status, int32_t radioId, bool muted) {
    MuteRadioResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    resp.set_muted(muted);
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_MUTE_RADIO_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendCancelRadioOperationsResponse(int32_t status, int32_t radioId) {
    CancelRadioOperationsResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_CANCEL_OPERATIONS_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendConfigureChannelSpacingResponse(int32_t status, int32_t radioId,
                                                       int32_t channelSpacing) {
    ConfigureChannelSpacingResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    resp.set_channel_spacing(channelSpacing);
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_CONFIGURE_CHANNEL_SPACING_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int RadioEndpoint::sendGetProgramListResponse(int32_t status, int32_t radioId,
                                              bool completed,
                                              const vector<StationInfoStruct> &stationInfos) {
    GetProgramListResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_radio_id(radioId);
    resp.set_completed(completed);
    for (vector<StationInfoStruct>::const_iterator it = stationInfos.begin();
         it != stationInfos.end(); ++it) {
        RadioStationInfo *rsi = resp.add_program_list();
        populateStationInfo(*it, rsi);
    }
    IoBuffer buf;
    mRouter->marshallProto(RADIO_MESSAGE_GET_PROGRAM_LIST_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

void RadioEndpoint::populateStationInfo(const StationInfoStruct &stationInfo,
                                        RadioStationInfo *proto) {
    proto->set_type(static_cast<RadioType>(stationInfo.type));
    proto->set_channel(stationInfo.channel);
    proto->set_sub_channel(stationInfo.subChannel);
    if (stationInfo.metaData) {
        populateMetaData(*stationInfo.metaData, proto->mutable_meta_data());
    }
}

void RadioEndpoint::populateMetaData(const StationMetaDataStruct &metaData,
                                     RadioStationMetaData *proto) {
    proto->set_audio_channels(metaData.audioChannels);
    proto->set_signal_quality(metaData.signalQuality);
    if (metaData.rds) {
        populateRdsData(*metaData.rds, proto->mutable_rds());
    }
    if (metaData.hdRadioData) {
        populateHdRadioData(*metaData.hdRadioData, proto->mutable_hd_station_info());
    }
}

void RadioEndpoint::populateRdsData(const RdsDataStruct& rdsData, RdsData* proto) {
  for (vector<int32_t>::const_iterator it = rdsData.alternativeFrequencyHz.begin();
       it != rdsData.alternativeFrequencyHz.end(); ++it) {
    proto->add_alternative_frequencies(*it);
  }
  proto->set_program_id(rdsData.programIdentification);
  proto->set_music_speech_switch(rdsData.musicSpeechSwitch);
  proto->set_program_service_name(rdsData.programService);
  proto->set_program_type(rdsData.programType);
  proto->set_program_type_name(rdsData.programTypeName);
  proto->set_radio_text(rdsData.radioText);
  proto->set_traffic_program_flag(rdsData.trafficProgramFlag);
  proto->set_traffic_announcement_flag(rdsData.trafficAnnouncementFlag);
}

void RadioEndpoint::populateHdRadioData(const HdRadioDataStruct& hdData,
    HdRadioStationInfo* proto) {
}

int RadioEndpoint::handleStepRequest(const StepChannelRequest &req) {
    return mRadioCallbacks->onStepChannelRequest(req.radio_id(), req.up(), req.skip_sub_channel());
}

int RadioEndpoint::handleSeekRequest(const SeekStationRequest &req) {
    return mRadioCallbacks->onSeekStationRequest(req.radio_id(), req.up(), req.skip_sub_channel());
}

int RadioEndpoint::handleScanRequest(const ScanStationsRequest &req) {
    return mRadioCallbacks->onScanStationsRequest(req.radio_id(), req.start(), req.up(),
        req.skip_sub_channel());
}

int RadioEndpoint::handleTuneRequest(const TuneToStationRequest &req) {
    return mRadioCallbacks->onTuneToStationRequest(req.radio_id(), req.channel(),
        req.sub_channel());
}

int RadioEndpoint::handleGetProgramListRequest(const GetProgramListRequest &req) {
    return mRadioCallbacks->onGetProgramListRequest(req.radio_id());
}

int RadioEndpoint::handleCancelRequest(const CancelRadioOperationsRequest &req) {
    return mRadioCallbacks->onCancelRadioOperationsRequest(req.radio_id());
}

int RadioEndpoint::handleConfigChannelSpacingRequest(const ConfigureChannelSpacingRequest &req) {
    return mRadioCallbacks->onConfigureChannelSpacingRequest(req.radio_id(), req.channel_spacing());
}

int RadioEndpoint::handleGetTrafficUpdateRequest(const GetTrafficUpdateRequest &req) {
    return mRadioCallbacks->onGetTrafficUpdateRequest(req.radio_id());
}

int RadioEndpoint::handleOnMuteRadioRequest(const MuteRadioRequest &req) {
    return mRadioCallbacks->onMuteRadioRequest(req.radio_id(), req.mute());
}

int RadioEndpoint::handleRadioSourceRequest(const RadioSourceRequest &req) {
    return mRadioCallbacks->onRadioSourceRequest();
}

int RadioEndpoint::handleSelectActiveRadioRequest(const SelectActiveRadioRequest &req) {
    return mRadioCallbacks->onSelectActiveRadioRequest(req.radio_id());
}
